﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Collections;
using System.Threading;


/*  EventWaitHandle class and the AutoResetEvent and ManualResetEvent derived classes 
    
    This example simulates a downloader application. A Downloader object is passed a list
    of Command objects which represent download information and a delegate value to be
    used in notifiying the caller upon completion of the whole operation.
    The Downloader object creates a coordinator thread and starts it. The coordinator thread then
    creates three worker threads, each one associated to a AutoResetEvent completion handle, and then issues a 
    WaitHandle.WaitAll(completionHandles) call. completionHandles is an array of AutoResetEvent objects.
    A AutoResetEvent object is also used in protecting access to the command list.
    Each worker thread, when starting execution, blocks its associated completion 
    WaitHandle, which is a AutoResetEvent, and signals the AutoResetEvent upon completion.
    When all completion handles are signalled the coordinator thread unblocks
    and notifies the caller for the completion of the download operation.*/
namespace Lessons
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            InitializeComponent();
            synContext = WindowsFormsSynchronizationContext.Current;
        }

        SynchronizationContext synContext;
        Random random = new Random();
        int jobCount = 0;


        ArrayList CreateCommandList()
        {
            int count = random.Next(8, 20);
            ArrayList list = new ArrayList();

            for (int i = 0; i < count; i++)
                list.Add(new Command());

            return list;
        }

        void CompletionCallBack(Downloader d)
        {
            synContext.Post(SynchronizedMethod, d);
        }

        void SynchronizedMethod(object state)
        {
            textBox1.Text += state.ToString() + Environment.NewLine;            

            ((IDisposable)state).Dispose();
            jobCount--;
        }

        private void button1_Click(object sender, EventArgs e)
        {
            Downloader d = new Downloader(CreateCommandList(), CompletionCallBack);
            jobCount++;
            d.Execute();
        }

        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = jobCount > 0;

            if (e.Cancel)
                MessageBox.Show("Please wait! A thread is still executed...");
        }
    }
}






namespace Lessons
{

    #region A Disposable base class
    public class Disposable : IDisposable
    {
        private bool disposed = false;


        private void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                    DisposeManagedResources();

                DisposeUnmanagedResources();

                disposed = true;
            }
        }

        protected virtual void DisposeManagedResources()
        {
        }

        protected virtual void DisposeUnmanagedResources()
        {
        }

        ~Disposable()
        {
            Dispose(false);
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        public bool IsDisposed
        {
            get { return disposed; }
        }

    }
    #endregion
    

    public class Command
    {
        static private Random random = new Random();
        private int dataSize = random.Next(3, 15);

        public void Execute()
        {
            int size = dataSize;
            while (size > 0)
            {
                size--;
                Thread.Sleep(50);
            }
        }

        public int DataSize { get { return dataSize; } }
    }



    public delegate void CompletionDelegate(Downloader d);



    public class Downloader : Disposable
    {
        static private int counter = 0;

        private int id = counter++;
        private int totalDataSize = 0;
        private int totalCommands = 0;
        private ArrayList commandList;
        private CompletionDelegate completionCallBack;

        private AutoResetEvent[] completionHandles = new AutoResetEvent[]
        {
            new AutoResetEvent(true),
            new AutoResetEvent(true),
            new AutoResetEvent(true)
        };

        private AutoResetEvent listProtector = new AutoResetEvent(true);



        private bool GetNextCommand(ref Command cmd)
        {
            cmd = null;

            if (listProtector.WaitOne())
            {
                try
                {                    
                    if (commandList.Count > 0)
                    {
                        cmd = (Command)commandList[0];
                        commandList.Remove(cmd);
                    }                    
                }
                finally
                {
                    listProtector.Set();
                }
            }

            return cmd != null;
        }
        
        private void CoordinatorThreadProc()
        {
            Thread worker;

            foreach (object completionHandle in completionHandles)
            {
                worker = new Thread(WorkerThreadProc);
                worker.Start(completionHandle);
            }

            Thread.Sleep(500);

            WaitHandle.WaitAll(completionHandles);

            completionCallBack(this);
        }

        private void WorkerThreadProc(object info)
        {
            AutoResetEvent completionHandle = (AutoResetEvent)info;
            completionHandle.WaitOne();
            try
            {
                Command cmd = null;
                while (GetNextCommand(ref cmd))
                    cmd.Execute();
            }
            finally
            {
                completionHandle.Set();
            }
        }

        protected override void DisposeUnmanagedResources()
        {
            if (!IsDisposed)
            {
                foreach (WaitHandle wh in completionHandles)
                    wh.Close();

                listProtector.Close();
            }
        }

        public Downloader(ArrayList CommandList, CompletionDelegate CompletionCallBack)
        {
            commandList = CommandList;
            completionCallBack = CompletionCallBack;

            totalCommands = commandList.Count;

            foreach (Command cmd in commandList)
                totalDataSize += cmd.DataSize;
        }


        public void Execute()
        {
            if (commandList.Count > 0)
            {
                Thread coordinator = new Thread(CoordinatorThreadProc);
                coordinator.Start();
            }
        }


        public override string ToString()
        {
            return "Job ID: " + id.ToString() +
                   ", Commands: " + totalCommands.ToString() +
                   ", Total DataSize: " + totalDataSize.ToString();
        }        
        

    }

}